/***************************************************************************
                          icemc.cpp  -  description
                             -------------------
    begin                :   17 22:11:56 EEST 2002
    copyright            : (C) 2002 by Vadim A. Khohlov
    email                : xvadim@newmail.ru
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>

#include <qaccel.h>
#include <qdir.h>

#include "icemc.h"
#include "qcentralwidget.h"
#include "filesave.xpm"
#include "fileopen.xpm"
#include "icewm.xpm"
#include "editcut.xpm"
#include "editcopy.xpm"
#include "editpaste.xpm"
#include "exit.xpm"

IcemcApp::IcemcApp():
icewmCfgDir(getenv("ICEWM_PRIVCFG")),
programName(tr("IceWM's menu/toolbar editor " VERSION))
{
  ///////////////////////////////////////////////////////////////////
  // call inits to invoke all other construction parts
  initActions();
  initMenuBar();
  initToolBar();
  //initStatusBar();

  if(icewmCfgDir.isNull())
    icewmCfgDir = QDir::homeDirPath() + "/.icewm";
  
  centralWidget = new QCentralWidget(this, "centralWidget", icewmCfgDir);
  setCentralWidget(centralWidget);
  
  if(qApp->argc() > 1)
	menuFileName = qApp->argv()[1];
  else
    menuFileName = icewmCfgDir + "/menu";
  centralWidget->loadMainMenuFromFile(menuFileName);
  setCaption(menuFileName + " - " + programName);
  fileSave->setEnabled(false);
  fileSaveAs->setEnabled(false);
  viewToolBar->setOn(true);
  viewStatusBar->setOn(true);
}

IcemcApp::~IcemcApp()
{
}

/** initializes all QActions of the application */
void IcemcApp::initActions()
{

  QPixmap openIcon(fileopen), saveIcon(filesave), icewmIcon(icewm);
  QPixmap cutIcon(editcut), copyIcon(editcopy), pasteIcon(editpaste);
  QPixmap exitIcon(exit_xpm);

  fileOpen = new QAction(tr("Open File"), openIcon,
		  tr("&Open..."),QAccel::stringToKey(tr("Ctrl+O")), this);
  fileOpen->setStatusTip(tr("Opens an existing document"));
  fileOpen->setWhatsThis(tr("Open File\n\nOpens an existing document"));
  connect(fileOpen, SIGNAL(activated()), this, SLOT(slotFileOpen()));

  fileSave = new QAction(tr("Save File"), saveIcon,
		  tr("&Save"), QAccel::stringToKey(tr("Ctrl+S")), this);
  fileSave->setStatusTip(tr("Saves the actual document"));
  fileSave->setWhatsThis(tr("Save File.\n\nSaves the actual document"));
  connect(fileSave, SIGNAL(activated()), this, SLOT(slotFileSave()));

  fileSaveAs = new QAction(tr("Save File As"), tr("Save &as..."), 0, this);
  fileSaveAs->setStatusTip(tr("Saves the actual document under a new filename"));
  fileSaveAs->setWhatsThis(tr("Save As\n\nSaves the actual document under a new filename"));
  connect(fileSaveAs, SIGNAL(activated()), this, SLOT(slotFileSave()));
  
  fileQuit = new QAction(tr("Exit"), exitIcon, tr("E&xit"),
		  QAccel::stringToKey(tr("Ctrl+Q")), this);
  fileQuit->setStatusTip(tr("Quits the application"));
  fileQuit->setWhatsThis(tr("Exit\n\nQuits the application"));
  connect(fileQuit, SIGNAL(activated()), this, SLOT(slotFileQuit()));

  fileRestartIce = new QAction(tr("Restart IceWM"), icewmIcon,
		  tr("&Restart IceWM"), QAccel::stringToKey(tr("Ctrl+R")), this);
  fileRestartIce->setStatusTip(tr("Restart the IceWM"));
  fileRestartIce->setWhatsThis(tr("Restart\n\nRestarts the IceWM"));
  connect(fileRestartIce, SIGNAL(activated()), this, SLOT(slotRestartIcewm()));
  
  editCut = new QAction(tr("Cut"), cutIcon, tr("&Cut"),
		  QAccel::stringToKey(tr("Ctrl+X")), this);
  editCut->setStatusTip(tr("Cuts the selected item into the clipboard"));
  editCut->setWhatsThis(tr("Cut\n\nCuts the selected item into the clipboard"));
  connect(editCut, SIGNAL(activated()), this, SLOT(slotCut()));
  
  editCopy = new QAction(tr("Copy"), copyIcon, tr("C&opy"),
		  QAccel::stringToKey(tr("Ctrl+C")), this);
  editCopy->setStatusTip(tr("Copies the selected item into the clipboard"));
  editCopy->setWhatsThis(tr("Copy\n\nCopies the selected item into the clipboard"));
  connect(editCopy, SIGNAL(activated()), this, SLOT(slotCopy()));
  
  editPaste = new QAction(tr("Paste"), pasteIcon, tr("&Paste"),
		  QAccel::stringToKey(tr("Ctrl+V")), this);
  editPaste->setStatusTip(tr("Pastes the clipboard into the menu"));
  editPaste->setWhatsThis(tr("Paste\n\nPastes the clipboard into the menu"));
  connect(editPaste, SIGNAL(activated()), this, SLOT(slotPaste()));
  
  viewToolBar = new QAction(tr("Toolbar"), tr("Tool&bar"), 0, this, 0, true);
  viewToolBar->setStatusTip(tr("Enables/disables the toolbar"));
  viewToolBar->setWhatsThis(tr("Toolbar\n\nEnables/disables the toolbar"));
  connect(viewToolBar, SIGNAL(toggled(bool)), this, SLOT(slotViewToolBar(bool)));

  viewStatusBar = new QAction(tr("Statusbar"), tr("&Statusbar"), 0, this, 0, true);
  viewStatusBar->setStatusTip(tr("Enables/disables the statusbar"));
  viewStatusBar->setWhatsThis(tr("Statusbar\n\nEnables/disables the statusbar"));
  connect(viewStatusBar, SIGNAL(toggled(bool)), this, SLOT(slotViewStatusBar(bool)));

  helpAboutApp = new QAction(tr("About"), tr("&About..."), 0, this);
  helpAboutApp->setStatusTip(tr("About the application"));
  helpAboutApp->setWhatsThis(tr("About\n\nAbout the application"));
  connect(helpAboutApp, SIGNAL(activated()), this, SLOT(slotHelpAbout()));
}

void IcemcApp::initMenuBar()
{
  ///////////////////////////////////////////////////////////////////
  // MENUBAR

  ///////////////////////////////////////////////////////////////////
  // menuBar entry fileMenu
  fileMenu=new QPopupMenu();
  fileOpen->addTo(fileMenu);
  fileMenu->insertSeparator();
  fileSave->addTo(fileMenu);
  fileSaveAs->addTo(fileMenu);
  fileMenu->insertSeparator();
  fileRestartIce->addTo(fileMenu);
  fileMenu->insertSeparator();
  fileQuit->addTo(fileMenu);

  editMenu = new QPopupMenu();
  editCut->addTo(editMenu);
  editCopy->addTo(editMenu);
  editPaste->addTo(editMenu);

  ///////////////////////////////////////////////////////////////////
  // menuBar entry viewMenu
  viewMenu=new QPopupMenu();
  viewMenu->setCheckable(true);
  viewToolBar->addTo(viewMenu);
  viewStatusBar->addTo(viewMenu);

  ///////////////////////////////////////////////////////////////////
  // menuBar entry helpMenu
  helpMenu=new QPopupMenu();
  helpAboutApp->addTo(helpMenu);

  ///////////////////////////////////////////////////////////////////
  // MENUBAR CONFIGURATION
  menuBar()->insertItem(tr("&File"), fileMenu);
  menuBar()->insertItem(tr("&Edit"), editMenu);
  menuBar()->insertItem(tr("&View"), viewMenu);
  menuBar()->insertSeparator();
  menuBar()->insertItem(tr("&Help"), helpMenu);
}

void IcemcApp::initToolBar()
{
  ///////////////////////////////////////////////////////////////////
  // TOOLBAR
  fileToolbar = new QToolBar(this, "file operations");
  fileOpen->addTo(fileToolbar);
  fileSave->addTo(fileToolbar);
  fileRestartIce->addTo(fileToolbar);
  fileToolbar->addSeparator();
  editCut->addTo(fileToolbar);
  editCopy->addTo(fileToolbar);
  editPaste->addTo(fileToolbar);
  fileToolbar->addSeparator();
  QWhatsThis::whatsThisButton(fileToolbar);
  fileToolbar->addSeparator();
  fileQuit->addTo(fileToolbar);
}

/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION
/////////////////////////////////////////////////////////////////////

void IcemcApp::slotFileOpen()
{
  statusBar()->message(tr("Opening file..."));
  centralWidget->checkDirty();
  menuFileName = QFileDialog::getOpenFileName(icewmCfgDir, 0,this);
  if (!menuFileName.isEmpty())
  {
    setCaption(menuFileName + " - " + programName);
    QString message=tr("Loaded document: ")+menuFileName;
	centralWidget->loadMainMenuFromFile(menuFileName);
    statusBar()->message(message, 2000);
	fileSave->setEnabled(false);
	fileSaveAs->setEnabled(false);
  }
  else
  {
    statusBar()->message(tr("Opening aborted"), 2000);
  }
}

void IcemcApp::slotFileSave()
{
  statusBar()->message(tr("Saving file..."));
  if(!menuFileName.isEmpty())
  {
  	centralWidget->saveMenu();
    setCaption(menuFileName + " - " + programName);
	fileSave->setEnabled(false);
	fileSaveAs->setEnabled(false);
  } /*   if(!menuFileName.isEmpty()) */
  else
	slotFileSaveAs();
  statusBar()->message(tr("Ready."));
}

void IcemcApp::slotFileSaveAs()
{
  statusBar()->message(tr("Saving file under new filename..."));
  menuFileName = QFileDialog::getSaveFileName(icewmCfgDir, 0, this);
  if (!menuFileName.isEmpty())
  {
	centralWidget->saveMenuAs(menuFileName);
    setCaption(menuFileName + " - " + programName);
	fileSave->setEnabled(false);
	fileSaveAs->setEnabled(false);
  } /*   if (!menuFileName.isEmpty()) */
  else
    statusBar()->message(tr("Saving aborted"), 2000);

  statusBar()->message(tr("Ready."));
}

void IcemcApp::slotFileQuit()
{
  statusBar()->message(tr("Exiting application..."));
  ///////////////////////////////////////////////////////////////////
  // exits the Application
  centralWidget->checkDirty();
  qApp->quit();
  statusBar()->message(tr("Ready."));
}

void IcemcApp::slotViewToolBar(bool toggle)
{
  statusBar()->message(tr("Toggle toolbar..."));
  ///////////////////////////////////////////////////////////////////
  // turn Toolbar on or off

  if (toggle== false)
  {
    fileToolbar->hide();
  }
  else
  {
    fileToolbar->show();
  };

  statusBar()->message(tr("Ready."));
}

void IcemcApp::slotViewStatusBar(bool toggle)
{
  statusBar()->message(tr("Toggle statusbar..."));
  ///////////////////////////////////////////////////////////////////
  //turn Statusbar on or off

  if (toggle == false)
  {
    statusBar()->hide();
  }
  else
  {
    statusBar()->show();
  }

  statusBar()->message(tr("Ready."));
}

void IcemcApp::slotHelpAbout()
{
  QMessageBox::about(this,tr("About..."),
   tr("Menu/toolbar editor for IceWM\nVersion " VERSION "\n(c) 2002-2004 by Vadim A. Khohlov") );
}

/** No descriptions */
void IcemcApp::slotRestartIcewm()
{
	system("killall -HUP icewm");
} /* void IcemcApp::slotRestartIcewm() */


/**
* Called by centralwidget after changing the content.
* Add '*' into caption.
*/
void
IcemcApp::contentChanged()
{
	setCaption(menuFileName + " [*] - " + programName);
	fileSave->setEnabled(true);
	fileSaveAs->setEnabled(true);
} /* IcemcApp ::contentChanged() */

/**
* Called by centralwidget after changing the menuFile.
*/
void
IcemcApp::menuFileChanged(const QString &mFileName)
{
	menuFileName = mFileName;
	setCaption(menuFileName + " - " + programName);
	fileSave->setEnabled(false);
	fileSaveAs->setEnabled(false);
} /* IcemcApp ::contentChanged() */

/** paste a text from clipboard AFTER selected item */
void
IcemcApp::slotPaste()
{
	centralWidget->paste();
} /* IcemcApp ::slotPaste() */

/** copy a selected item to clipboard */
void
IcemcApp::slotCopy()
{
	centralWidget->copy();
} /* IcemcApp ::slotCopy() */

/** cut a selected item to clipboard */
void
IcemcApp::slotCut()
{
	centralWidget->cut();
} /* IcemcApp ::slotCut() */
